<?php

namespace P2c2p\P2c2pPayment\Service;

use Jose\Component\Core\AlgorithmManager;
use Jose\Component\Encryption\Algorithm\KeyEncryption\RSAOAEP;
use Jose\Component\Encryption\Algorithm\ContentEncryption\A256GCM;
use Jose\Component\Encryption\Compression\CompressionMethodManager;
use Jose\Component\Encryption\Compression\Deflate;
use Jose\Component\Encryption\JWEBuilder;
use Jose\Component\KeyManagement\JWKFactory;
use Jose\Component\Encryption\Serializer\CompactSerializer;
use Jose\Component\Signature\Serializer\CompactSerializer as JWSCompactSerializer;
use Jose\Component\Signature\Serializer\JWSSerializerManager;
use Jose\Component\Encryption\Serializer\JWESerializerManager;

use Jose\Component\Signature\Algorithm\PS256;
use Jose\Component\Signature\JWSBuilder;

use Jose\Component\Signature\JWSVerifier;
use Jose\Component\Encryption\JWEDecrypter;

use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;


use P2c2p\P2c2pPayment\Model\Config\Backend\FilePathHelper;
use Psr\Log\LoggerInterface as PsrLoggerInterface;

class PaymentService
{
    private $curl;
    private $objConfigSettings;

    protected $filePathHelper;
    protected $_configSettings;
    protected $_moduleConfigSettings;

    public function __construct(
        Curl $curl,
        ScopeConfigInterface $configSettings,
        FilePathHelper $filePathHelper
    ) {
        $this->_configSettings = $configSettings;
        $this->_moduleConfigSettings = $configSettings->getValue(
            "payment/p2c2ppayment",
            ScopeInterface::SCOPE_STORE
        );

        $this->curl = $curl;
        $this->filePathHelper = $filePathHelper;
    }

    protected function getFilePath($relativePath)
    {
        return $this->filePathHelper->getFullPath($relativePath);
    }

    protected function writeToCustomLog($message)
    {
        $logDir = BP . '/var/log/p2cp2/';
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $logFile = $logDir . 'debug.log';
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[{$timestamp}] {$message}" . PHP_EOL;
        
        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }

    public function v4_payment_action_jwe ($content, $kid_2c2p = '1', $kid_merchant = '1') {
        $private_key_password_merchant = $this->_moduleConfigSettings["merchant_key_password"];
        // $private_key_password_merchant = '2c2p';
        $merchant_key = $this->_moduleConfigSettings["merchant_key"];
        $public_cert = $this->_moduleConfigSettings["public_cert"];
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        // $psrLogger->debug('merchant_key: ' . print_r($merchant_key,1));
        // $psrLogger->debug('public_cert: ' . print_r($public_cert,1));

        $public_key_2c2p = $this->getFilePath($public_cert);
        $private_key_merchant = $this->getFilePath($merchant_key);
        
        // The key encryption algorithm manager with the A256KW algorithm.
        $keyEncryptionAlgorithmManager = new AlgorithmManager([
            new RSAOAEP(),
        ]);
    
        // The content encryption algorithm manager with the A256CBC-HS256 algorithm.
        $contentEncryptionAlgorithmManager = new AlgorithmManager([
            new A256GCM(),
        ]);
    
        // The compression method manager with the DEF (Deflate) method.
        $compressionMethodManager = new CompressionMethodManager([
            new Deflate(),
        ]);
    
        // We instantiate our JWE Builder.
        $jweBuilder = new JWEBuilder($keyEncryptionAlgorithmManager, $contentEncryptionAlgorithmManager, $compressionMethodManager);

        // Our key.
        $jwk = JWKFactory::createFromCertificateFile($public_key_2c2p,[
            'use' => 'enc',
            'kid' => $kid_2c2p
        ]);
        // $psrLogger->debug('jwk: ' . print_r($jwk,1));
    
        $jwe = $jweBuilder
        ->create()              // We want to create a new JWE
        ->withPayload($content) // We set the payload
        ->withSharedProtectedHeader([
            'alg' => 'RSA-OAEP', // Key Encryption Algorithm
            'enc' => 'A256GCM', // Content Encryption Algorithm
            // 'zip' => 'DEF'            // We enable the compression (irrelevant as the payload is small, just for the example).
            'kid' => $kid_2c2p
        ])
        ->addRecipient($jwk)    // We add a recipient (a shared key or public key).
        ->build();              // We build it
    
        $serializer = new CompactSerializer(); // The serializer
        $token = $serializer->serialize($jwe);;
        
        // Signature
    
        // The algorithm manager with the HS256 algorithm.
        $algorithmManager = new AlgorithmManager([
            new PS256(),
        ]);
    
        $privateKey = JWKFactory::createFromKeyFile(
            $private_key_merchant,
            $private_key_password_merchant,
            [
                'use' => 'sig'
            ]
        );
    
        // We instantiate our JWS Builder.
        $jwsBuilder = new JWSBuilder($algorithmManager);
    
        $jws = $jwsBuilder
        ->create() // We want to create a new JWS
        ->withPayload($token) // We set the payload
        ->addSignature($privateKey, ['alg' => 'PS256' , 'kid' => $kid_merchant]) // We add a signature with a simple protected header
        ->build();
    
        $jws_serializer = new JWSCompactSerializer();
        $jws_token = $jws_serializer->serialize($jws, 0); 
        // $psrLogger->debug('jws_token: ' . print_r($jws_token,1));
        return $jws_token;
    }

    function v4_payment_action_verify ($content) {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        $public_cert = $this->_moduleConfigSettings["public_cert"];
        $public_key_2c2p = $this->getFilePath($public_cert);
        
        // $psrLogger->debug('v4_payment_action_verify!');
        // $psrLogger->debug('content: ' . print_r($content,1));
        // $psrLogger->debug('public_cert: ' . print_r($public_cert,1));
        // $psrLogger->debug('public_key_2c2p: ' . print_r($public_key_2c2p,1));

        // The algorithm manager with the HS256 algorithm.
        $algorithmManager = new AlgorithmManager([
          new PS256(),
        ]);
      
        // We instantiate our JWS Verifier.
        $jwsVerifier = new JWSVerifier(
          $algorithmManager
        );
      
      
        // Our key.
        $jwk = JWKFactory::createFromCertificateFile(
          $public_key_2c2p, // The filename
            [
              'use' => 'sig'         // Additional parameters
            ]
        );
      
        // The serializer manager. We only use the JWS Compact Serialization Mode.
        $serializerManager = new JWSSerializerManager([
          new JWSCompactSerializer(),
        ]);
      
        // The input we want to check
        $token = $content;
      
        // We try to load the token.
        $jws = $serializerManager->unserialize($token);
      
        // We verify the signature. This method does NOT check the header.
        // The arguments are:
        // - The JWS object,
        // - The key,
        // - The index of the signature to check. See 
        $isVerified = $jwsVerifier->verifyWithKey($jws, $jwk, 0);
      
        if($isVerified) {
          return $jws->getPayload();
        }
      
        return $isVerified;
    }

    function v4_payment_action_decrypt ($content) {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        $psrLogger->debug('=== STARTING v4_payment_action_decrypt ===');
        
        // Write directly to custom log file for debugging
        $this->writeToCustomLog('=== STARTING v4_payment_action_decrypt ===');
        
        // Get configuration values
        $merchant_key = $this->_moduleConfigSettings["merchant_key"];
        $private_key_merchant = $this->getFilePath($merchant_key);
        $private_key_password_merchant = $this->_moduleConfigSettings["merchant_key_password"];
        
        // Log configuration values
        $psrLogger->debug('merchant_key config: ' . print_r($merchant_key, 1));
        $psrLogger->debug('private_key_merchant path: ' . print_r($private_key_merchant, 1));
        $psrLogger->debug('private_key_password_merchant: ' . print_r($private_key_password_merchant, 1));
        
        $this->writeToCustomLog('merchant_key config: ' . print_r($merchant_key, 1));
        $this->writeToCustomLog('private_key_merchant path: ' . print_r($private_key_merchant, 1));
        $this->writeToCustomLog('private_key_password_merchant: ' . print_r($private_key_password_merchant, 1));
        
        // Validate configuration
        if (empty($merchant_key)) {
            $psrLogger->error('ERROR: merchant_key configuration is empty');
            $this->writeToCustomLog('ERROR: merchant_key configuration is empty');
            throw new Exception('Merchant key not configured');
        }
        
        if (empty($private_key_password_merchant)) {
            $psrLogger->error('ERROR: merchant_key_password configuration is empty');
            $this->writeToCustomLog('ERROR: merchant_key_password configuration is empty');
            throw new Exception('Merchant key password not configured');
        }
        
        // Check if file exists
        if (!file_exists($private_key_merchant)) {
            $psrLogger->error('ERROR: Merchant key file not found at path: ' . $private_key_merchant);
            $this->writeToCustomLog('ERROR: Merchant key file not found at path: ' . $private_key_merchant);
            throw new Exception('Merchant key file not found: ' . $private_key_merchant);
        }
        
        // Check file permissions
        if (!is_readable($private_key_merchant)) {
            $psrLogger->error('ERROR: Merchant key file is not readable: ' . $private_key_merchant);
            $this->writeToCustomLog('ERROR: Merchant key file is not readable: ' . $private_key_merchant);
            throw new Exception('Merchant key file is not readable: ' . $private_key_merchant);
        }
        
        // Log file info
        $fileSize = filesize($private_key_merchant);
        $filePerms = substr(sprintf('%o', fileperms($private_key_merchant)), -4);
        $psrLogger->debug('Key file size: ' . $fileSize . ' bytes');
        $psrLogger->debug('Key file permissions: ' . $filePerms);
        
        $this->writeToCustomLog('Key file size: ' . $fileSize . ' bytes');
        $this->writeToCustomLog('Key file permissions: ' . $filePerms);
        
        // Read and log first few lines of the key file (for debugging)
        $keyContent = file_get_contents($private_key_merchant);
        $keyLines = explode("\n", $keyContent);
        $psrLogger->debug('Key file first 3 lines:');
        $this->writeToCustomLog('Key file first 3 lines:');
        for ($i = 0; $i < min(3, count($keyLines)); $i++) {
            $psrLogger->debug('Line ' . ($i + 1) . ': ' . trim($keyLines[$i]));
            $this->writeToCustomLog('Line ' . ($i + 1) . ': ' . trim($keyLines[$i]));
        }
        
        // Check if it looks like a valid key file
        if (strpos($keyContent, '-----BEGIN') === false) {
            $psrLogger->error('ERROR: Key file does not appear to be a valid PEM format (missing -----BEGIN)');
            $this->writeToCustomLog('ERROR: Key file does not appear to be a valid PEM format (missing -----BEGIN)');
            throw new Exception('Invalid key file format - not a valid PEM file');
        }

        // The key encryption algorithm manager with the A256KW algorithm.
        $keyEncryptionAlgorithmManager = new AlgorithmManager([
            new RSAOAEP(),
        ]);
      
        // The content encryption algorithm manager with the A256CBC-HS256 algorithm.
        $contentEncryptionAlgorithmManager = new AlgorithmManager([
            new A256GCM(),
        ]);
      
        // The compression method manager with the DEF (Deflate) method.
        $compressionMethodManager = new CompressionMethodManager([
            new Deflate(),
        ]);
      
        // We instantiate our JWE Decrypter.
        $jweDecrypter = new JWEDecrypter(
            $keyEncryptionAlgorithmManager,
            $contentEncryptionAlgorithmManager,
            $compressionMethodManager
        );
      
        // Try to create the private key with detailed error handling
        $psrLogger->debug('Attempting to create private key from file...');
        $this->writeToCustomLog('Attempting to create private key from file...');
        try {
            $privateKey = JWKFactory::createFromKeyFile(
                $private_key_merchant,
                $private_key_password_merchant,
                [
                    'use' => 'enc',
                ]
            );
            $psrLogger->debug('SUCCESS: Private key created successfully');
            $this->writeToCustomLog('SUCCESS: Private key created successfully');
        } catch (Exception $e) {
            $psrLogger->error('ERROR: Failed to create private key from file: ' . $e->getMessage());
            $psrLogger->error('Exception type: ' . get_class($e));
            $psrLogger->error('Exception trace: ' . $e->getTraceAsString());
            $this->writeToCustomLog('ERROR: Failed to create private key from file: ' . $e->getMessage());
            $this->writeToCustomLog('Exception type: ' . get_class($e));
            $this->writeToCustomLog('Exception trace: ' . $e->getTraceAsString());
            throw new Exception('Unable to decrypt the key: ' . $e->getMessage());
        } catch (Error $e) {
            $psrLogger->error('FATAL ERROR: Failed to create private key from file: ' . $e->getMessage());
            $psrLogger->error('Error type: ' . get_class($e));
            $psrLogger->error('Error trace: ' . $e->getTraceAsString());
            $this->writeToCustomLog('FATAL ERROR: Failed to create private key from file: ' . $e->getMessage());
            $this->writeToCustomLog('Error type: ' . get_class($e));
            $this->writeToCustomLog('Error trace: ' . $e->getTraceAsString());
            throw new Exception('Unable to decrypt the key: ' . $e->getMessage());
        }
      
        // The serializer manager. We only use the JWE Compact Serialization Mode.
        $serializerManager = new JWESerializerManager([
            new CompactSerializer(),
        ]);
      
        // The input we want to decrypt
        $token = $content;
        $psrLogger->debug('Token length: ' . strlen($token));
        $psrLogger->debug('Token preview (first 100 chars): ' . substr($token, 0, 100));
        
        $this->writeToCustomLog('Token length: ' . strlen($token));
        $this->writeToCustomLog('Token preview (first 100 chars): ' . substr($token, 0, 100));
      
        // We try to load the token.
        $psrLogger->debug('Attempting to unserialize JWE token...');
        $this->writeToCustomLog('Attempting to unserialize JWE token...');
        try {
            $jwe = $serializerManager->unserialize($token);
            $psrLogger->debug('SUCCESS: JWE token unserialized successfully');
            $this->writeToCustomLog('SUCCESS: JWE token unserialized successfully');
        } catch (Exception $e) {
            $psrLogger->error('ERROR: Failed to unserialize JWE token: ' . $e->getMessage());
            $this->writeToCustomLog('ERROR: Failed to unserialize JWE token: ' . $e->getMessage());
            throw new Exception('Invalid JWE token format: ' . $e->getMessage());
        }
      
        // We decrypt the token. This method does NOT check the header.
        $psrLogger->debug('Attempting to decrypt JWE token...');
        $this->writeToCustomLog('Attempting to decrypt JWE token...');
        try {
            $success = $jweDecrypter->decryptUsingKey($jwe, $privateKey, 0);
            $psrLogger->debug('Decryption result: ' . ($success ? 'SUCCESS' : 'FAILED'));
            $this->writeToCustomLog('Decryption result: ' . ($success ? 'SUCCESS' : 'FAILED'));
            
            if($success) {
                $payload = $jwe->getPayload();
                $psrLogger->debug('SUCCESS: Decryption completed, payload length: ' . strlen($payload));
                $this->writeToCustomLog('SUCCESS: Decryption completed, payload length: ' . strlen($payload));
                return $payload;
            } else {
                $psrLogger->error('ERROR: Decryption failed - success returned false');
                $this->writeToCustomLog('ERROR: Decryption failed - success returned false');
                return false;
            }
        } catch (Exception $e) {
            $psrLogger->error('ERROR: Exception during decryption: ' . $e->getMessage());
            $psrLogger->error('Exception type: ' . get_class($e));
            $psrLogger->error('Exception trace: ' . $e->getTraceAsString());
            $this->writeToCustomLog('ERROR: Exception during decryption: ' . $e->getMessage());
            $this->writeToCustomLog('Exception type: ' . get_class($e));
            $this->writeToCustomLog('Exception trace: ' . $e->getTraceAsString());
            throw new Exception('Unable to decrypt the key: ' . $e->getMessage());
        } catch (Error $e) {
            $psrLogger->error('FATAL ERROR: Error during decryption: ' . $e->getMessage());
            $psrLogger->error('Error type: ' . get_class($e));
            $psrLogger->error('Error trace: ' . $e->getTraceAsString());
            $this->writeToCustomLog('FATAL ERROR: Error during decryption: ' . $e->getMessage());
            $this->writeToCustomLog('Error type: ' . get_class($e));
            $this->writeToCustomLog('Error trace: ' . $e->getTraceAsString());
            throw new Exception('Unable to decrypt the key: ' . $e->getMessage());
        }
        
        $psrLogger->debug('=== ENDING v4_payment_action_decrypt ===');
        $this->writeToCustomLog('=== ENDING v4_payment_action_decrypt ===');
    }

    /*public function sendPaymentRequest($paymentRequest)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        $psrLogger->debug('sendPaymentRequest');
        $merchant_cert = $this->_moduleConfigSettings["merchant_cert"];
        $merchant_key = $this->_moduleConfigSettings["merchant_key"];
        $public_cert = $this->_moduleConfigSettings["public_cert"];
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        $psrLogger->debug('merchant_cert: ' . print_r($merchant_cert,1));
        $psrLogger->debug('merchant_key: ' . print_r($merchant_key,1));
        $psrLogger->debug('public_cert: ' . print_r($public_cert,1));

        $senderPrivateKeyPath = $this->getFilePath($merchant_key);
        $receiverPublicCertPath = $this->getFilePath($public_cert);
        $senderPrivateKeyPassword = '';
        $psrLogger->debug('senderPrivateKeyPath: ' . print_r($senderPrivateKeyPath,1));
        $psrLogger->debug('receiverPublicCertPath: ' . print_r($receiverPublicCertPath,1));

        // $receiverPublicCertPath = 'C:/cert/sandbox-jwt-2c2p.demo.2.1(public).cer';
        // $senderPrivateKeyPath = 'C:/cert/Merchant12345.pfx';
        // $senderPrivateKeyPassword = 'm12345';

        // Load the receiver's public key
        $receiverPublicKey = openssl_pkey_get_public(file_get_contents($receiverPublicCertPath));
        $psrLogger->debug('receiverPublicKey: ' . print_r($receiverPublicKey,1));

        // Load the sender's private key
        $pkcs12 = file_get_contents($senderPrivateKeyPath);
        $psrLogger->debug('pkcs12: ' . print_r($pkcs12,1));
        $psrLogger->debug('certs: ' . print_r($certs,1));
        $psrLogger->debug('senderPrivateKeyPassword: ' . print_r($senderPrivateKeyPassword,1));
        openssl_pkcs12_read($pkcs12, $certs, $senderPrivateKeyPassword);
        $senderPrivateKey = openssl_pkey_get_private($certs['pkey']);
        $psrLogger->debug('senderPrivateKey: ' . print_r($senderPrivateKey,1));

        // Create the JWE
        $jweFactory = new JWEBuilderFactory();
        $jweBuilder = $jweFactory->create(['RSA-OAEP'], ['A256GCM']);
        $jwe = $jweBuilder
            ->create()
            ->withPayload($paymentRequest)
            ->withSharedProtectedHeader([
                'alg' => 'RSA-OAEP',
                'enc' => 'A256GCM'
            ])
            ->addRecipient(new JWK(['kty' => 'RSA', 'n' => base64_encode($receiverPublicKey), 'e' => 'AQAB']))
            ->build();

        $jweSerializer = new CompactSerializer();
        $jweRequest = $jweSerializer->serialize($jwe);

        // Create the JWS
        $jwsFactory = new JWSBuilderFactory();
        $jwsBuilder = $jwsFactory->create(['PS256']);
        $jws = $jwsBuilder
            ->create()
            ->withPayload($jweRequest)
            ->addSignature(new JWK(['kty' => 'RSA', 'd' => base64_encode($senderPrivateKey)]), ['alg' => 'PS256'])
            ->build();

        $jwsSerializer = new JWSSerializer();
        $jwsRequest = $jwsSerializer->serialize($jws);

        // Send the request using Magento's Curl class
        $this->curl->addHeader("Content-Type", "application/*+json");
        $this->curl->addHeader("Accept", "text/plain");
        $this->curl->post("api_endpoint", $jwsRequest);

        return $this->curl->getBody();
    }*/
}
