<?php

/*
 * Created by 2C2P
 * Date 19 June 2017
 * PaymentMethod is base class / entry point for 2c2p plugin.
 */

namespace P2c2p\P2c2pPayment\Model;
use Magento\Payment\Model\Method\AbstractMethod;
use Magento\Framework\DataObject;
use Magento\Framework\App\ObjectManager;
use Magento\Payment\Model\InfoInterface;
use Magento\Framework\Validator\Exception;
use Magento\Sales\Model\Order\Payment\Transaction;
use Psr\Log\LoggerInterface as PsrLoggerInterface;

use P2c2p\P2c2pPayment\Service\PaymentService;
use Magento\Framework\HTTP\Client\Curl;

class PaymentMethod extends AbstractMethod
{
    protected $_code = "p2c2ppayment";
    protected $_isInitializeNeeded = true;
    protected $_isGateway = true;

    protected $_canAuthorize = true;
    protected $_canCapture = true;
    protected $_canUseInternal = true;
    protected $_canRefund = true;
    protected $_canVoid = true;
    protected $_canRefundInvoicePartial = true;
    protected $_canUseCheckout = true;
    protected $_canUseForMultishipping = false;

    protected $paymentService;
    protected $curl;

    private $objConfigSettings;
    private $merchant_id;
    private $secretKey;
    private $processType;
    private $invoiceNo;
    private $version;
    private $hash;

    private $url = "https://demo2.2c2p.com/2C2PFrontend/PaymentAction/2.0/action";

    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        PaymentService $paymentService,
        Curl $curl
    ) {
        // Call parent constructor to initialize necessary dependencies
        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger
        );

        $this->curl = $curl;
        $this->paymentService = $paymentService;
    }

    protected function writeToCustomLog($message)
    {
        $logDir = BP . '/var/log/p2cp2/';
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $logFile = $logDir . 'debug.log';
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[{$timestamp}] {$message}" . PHP_EOL;
        
        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }

    public function log($data)
    {
    }

    //Set additional data and session object and use it further process.
    public function assignData(DataObject $data)
    {
        parent::assignData($data);

		$objectManager = \Magento\Framework\App\ObjectManager::getInstance();
		$catalogSession = $objectManager->create('\Magento\Catalog\Model\Session');
        $customerSession = $objectManager->get(\Magento\Customer\Model\Session::class);

        // Check if customer is logged in
        if (!$customerSession->isLoggedIn()) {
            return $this; // Exit early if user is not logged in
        }

        $this->_logger->info('data getData: ' . print_r($data->getData(),1));
		if(isset($data)) {
            $additionalData = $data->getData('additional_data');
            if (!empty($additionalData) && !empty($additionalData['cardtoken'])) {
                $catalogSession->setTokenValue($additionalData['cardtoken']);
            }
		}

        return $this;
    }

    public function void(InfoInterface $payment)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class);
        $psrLogger->debug('void:');
        $this->writeToCustomLog('=== STARTING VOID OPERATION ===');
        // $this->log('in_void');
        $order = $payment->getOrder();
        $orderid = $order->getRealOrderId();
        $this->writeToCustomLog('Void order ID: ' . $orderid);
        // $this->log($orderid);

        $amount = $order->getGrandTotal();
        $this->writeToCustomLog('Void amount: ' . $amount);

        $status = $this->inquiry($orderid);
        $this->writeToCustomLog('Void inquiry status: ' . print_r($status, 1));

        if (is_null($status)) {
            // $this->log('in_void inquiry object is null / empty.');
            throw new Exception(__("Cannot query payment status."));
        }

        $res = null;
        if ($status['status'] == "A") {
            $res = $this->c2p_void($orderid);
        } elseif ($status['status'] == "S") {
            $res = $this->c2p_refund($orderid, $amount);
        } elseif ($status['status'] == "V") {
            // $this->log('in_void: payment was already voided.');
            return $this;
        } else {
            throw new Exception(
                __(
                    "Payment status (" .
                        $status['status'] .
                        ") is not valid to perform void/refund. Try again later."
                )
            );
        }

        if (is_null($res)) {
            // $this->log('in_void res object is null.');
            throw new Exception(__("Cannot perform void payment."));
        }

        // $this->log('in_void respCode=' . $res->respCode .', status='.$res->status);
        if ($res['respCode'] == "00" || $res['status'] == "V") {
            return $this;
        } else {
            throw new Exception(
                __("Cannot perform void payment. reason_code:" . $res['respCode'])
            );
        }
    }

    public function cancel(InfoInterface $payment)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class);
        // $this->log('in_cancel');
        $order = $payment->getOrder();
        $orderid = $order->getRealOrderId();
        // $this->log($orderid);

        $amount = $order->getGrandTotal();

        $status = $this->inquiry($orderid);

        if (is_null($status)) {
            // $this->log('in_void inquiry object is null / empty.');
            throw new Exception(__("Cannot query payment status."));
        }

        $res = null;
        if ($status['status'] == "A") {
            $res = $this->c2p_void($orderid);
            $psrLogger->debug('c2p_void res: ' . print_r($res,1));
        } elseif ($status['status'] == "S") {
            $res = $this->c2p_refund($orderid, $amount);
            $psrLogger->debug('c2p_refund res: ' . print_r($res,1));
        } elseif ($status['status'] == "V") {
            // $this->log('in_void: payment was already voided.');
            return $this;
        } else {
            throw new Exception(
                __(
                    "Payment status (" .
                        $status['status'] .
                        ") is not valid to perform void/refund. Try again later."
                )
            );
        }

        if (is_null($res)) {
            // $this->log('in_void res object is null.');
            throw new Exception(__("Cannot perform cancel payment."));
        }

        // $this->log('in_void respCode=' . $res->respCode .', status='.$res->status);
        if ($res['respCode'] == "00" || $res['status'] == "V") {
            return $this;
        } else {
            throw new Exception(
                __(
                    __("Cannot perform refund. reason_code:" . $res['respCode'])
                )
            );
        }
    }

    public function capture(InfoInterface $payment, $amount)
    {
        $this->loadsettings();
        $this->paymentAction = $this->objConfigSettings["payment_action"];
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class);
        $psrLogger->debug('capture!');
        if ($this->paymentAction == 'authorize') {
            $psrLogger->debug('exec capture!');
            $order = $payment->getOrder();
            $orderid = $order->getRealOrderId();
            $amount = $order->getGrandTotal();

            // $this->log($amount);
            $res = $this->c2p_capture($orderid, $amount);
            $psrLogger->debug('res: ' . print_r($res,1));

            if (is_null($res)) {
                throw new Exception(__("Cannot perform capture."));
            }

            if (
                $res['respCode'] == "00" ||
                $res['respCode'] == "32" || // Settlement/Capture not required
                $res['status'] == "S" ||
                $res['status'] == "SS" ||
                $res['status'] == "RS"
            ) {
                $order = $payment->getOrder();
                $billing = $order->getBillingAddress();
                $payment->setTransactionId($orderid)->setIsTransactionClosed(true);
                return $this;
            } else {
                throw new Exception(
                    __("Payment capture error. reason_code:" . $res['respCode'])
                );
            }
        }
    }

    public function refund(InfoInterface $payment, $amount)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class);
        $psrLogger->debug('refund:');
        $this->writeToCustomLog('=== STARTING REFUND OPERATION ===');
        $order = $payment->getOrder();
        $orderid = $order->getRealOrderId();
        $this->writeToCustomLog('Refund order ID: ' . $orderid);

        $total = $order->getGrandTotal();
        $this->writeToCustomLog('Refund amount: ' . $amount . ', Total: ' . $total);
        $status = $this->inquiry($orderid);
        $psrLogger->debug('status: ' . print_r($status,1));
        $this->writeToCustomLog('Refund inquiry status: ' . print_r($status, 1));

        //throw new \Magento\Framework\Validator\Exception(__('Cannot perform refund.'.$amount));

        if (is_null($status)) {
            // $this->log('in_void inquiry object is null / empty.');
            throw new Exception(__("Cannot query payment status."));
        }

        $res = null;
        if ($status['status'] == "A") {
            if ($amount < $total) {
                throw new Exception(
                    __(
                        "Transaction still pending capture. Cannot be refunded yet."
                    )
                );
            } else {
                $res = $this->c2p_void($orderid);
                $psrLogger->debug('c2p_void res: ' . print_r($res,1));
            }
        } elseif ($status['status'] == "S") {
            $res = $this->c2p_refund($orderid, $amount);
            $psrLogger->debug('c2p_refund res: ' . print_r($res,1));
        } elseif ($status['status'] == "V") {
            // $this->log('in_void: payment was already voided.');
            return $this;
        } else {
            throw new Exception(
                __(
                    "Payment status (" .
                        $status['status'] .
                        ") is not valid to perform void/refund. Try again later."
                )
            );
        }

        if (is_null($res)) {
            throw new Exception(__("Cannot perform refund."));
        }

        if ($res['respCode'] == "00" || $res['status'] == "V") {
            $transactionId = $payment->getParentTransactionId();

            if ($amount < $total) {
                $payment
                    ->setTransactionId(
                        $transactionId . "-" . Transaction::TYPE_REFUND
                    )
                    ->setParentTransactionId($transactionId)
                    ->setIsTransactionClosed(0)
                    ->setShouldCloseParentTransaction(0);
            } else {
                $payment
                    ->setTransactionId(
                        $transactionId . "-" . Transaction::TYPE_REFUND
                    )
                    ->setParentTransactionId($transactionId)
                    ->setIsTransactionClosed(1)
                    ->setShouldCloseParentTransaction(1);
            }

            return $this;
        } else {
            throw new Exception(
                __("Cannot perform refund. reason_code:" . $res['respCode'])
            );
        }
    }

    /**
     * Instantiate state and set it to state object
     * @param string $paymentAction
     * @param Varien_Object
     */
    public function initialize($paymentAction, $stateObject)
    {
        $stateObject->setState("Pending_2C2P");
        $stateObject->setStatus("Pending_2C2P");
        $stateObject->setIsNotified(false);
    }

    // 2c2p helper functions //////////////////////

    public function inquiry($invoiceNo)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class);
        $psrLogger->debug('inquiry');
        $psrLogger->debug('invoiceNo' . $invoiceNo);
        $responsePayment = $this->inq_paymentProcess($invoiceNo);
        $psrLogger->debug('responsePayment: ' . $responsePayment);
        $xml = simplexml_load_string($responsePayment);
        $resXml = json_decode(json_encode($xml), true);
        $psrLogger->debug('resXml: ' . print_r($resXml,1));
        return $resXml;
    }

    public function c2p_capture($invoiceNo, $amount)
    {
        // $this->log('in_c2p_capture');
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class);
        $psrLogger->debug('in_c2p_capture');
        $psrLogger->debug('invoiceNo' . $invoiceNo);
        $psrLogger->debug('amount' . $amount);
        $responsePayment = $this->cap_paymentProcess($invoiceNo, $amount);
        $psrLogger->debug('responsePayment: ' . $responsePayment);
        $xml = simplexml_load_string($responsePayment);
        $resXml = json_decode(json_encode($xml), true);
        $psrLogger->debug('resXml: ' . print_r($resXml,1));
        return $resXml;
    }

    public function c2p_refund($invoiceNo, $amount)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class);
        $psrLogger->debug('c2p_refund');
        $psrLogger->debug('invoiceNo' . $invoiceNo);
        $psrLogger->debug('amount' . $amount);
        $responsePayment = $this->ref_paymentProcess($invoiceNo, $amount);
        $psrLogger->debug('responsePayment: ' . $responsePayment);
        $xml = simplexml_load_string($responsePayment);
        $resXml = json_decode(json_encode($xml), true);
        $psrLogger->debug('resXml: ' . print_r($resXml,1));
        return $resXml;
    }

    public function c2p_void($invoiceNo)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class);
        $psrLogger->debug('c2p_void');
        $psrLogger->debug('invoiceNo' . $invoiceNo);
        $responsePayment = $this->vod_paymentProcess($invoiceNo);
        $psrLogger->debug('responsePayment: ' . $responsePayment);
        $xml = simplexml_load_string($responsePayment);
        $resXml = json_decode(json_encode($xml), true);
        $psrLogger->debug('resXml: ' . print_r($resXml,1));
        return $resXml;
    }

    function getPaymentGetwayRedirectUrl()
    {
        if ($this->objConfigSettings["mode"]) {
            return "https://demo2.2c2p.com/2C2PFrontend/PaymentActionV2/PaymentProcess.aspx";
        } else {
            return "https://t.2c2p.com/PaymentActionV2/PaymentProcess.aspx";
        }
    }

    public function loadsettings()
    {
        $objectManager = ObjectManager::getInstance();
        $configSettings = $objectManager->create(
            "Magento\Framework\App\Config\ScopeConfigInterface"
        );
        $this->objConfigSettings = $configSettings->getValue(
            "payment/p2c2ppayment",
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
    }

    //inquiry
    public function inq_setTheParameters($invoiceNo)
    {
        $this->merchant_id = $this->objConfigSettings["merchantId"];
        $this->secretKey = $this->objConfigSettings["secretKey"];
        $this->processType = "I";
        $this->invoiceNo = $invoiceNo;
        $this->version = "3.4";
        $stringToHash =
            $this->version .
            $this->merchant_id .
            $this->processType .
            $this->invoiceNo;
        $this->hash = strtoupper(
            hash_hmac("sha1", $stringToHash, $this->secretKey, false)
        ); //Compute hash value
    }

    public function inq_paymentProcess($invoiceNo)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        $psrLogger->debug('inq_paymentProcess');
        $this->writeToCustomLog('=== STARTING inq_paymentProcess ===');
        $this->loadsettings();

        //set up the parameters
        $this->inq_setTheParameters($invoiceNo);

        //Construct request message
        $paymentRequest = "<PaymentProcessRequest>
                <version>{$this->version}</version>
                <merchantID>{$this->merchant_id}</merchantID>
                <processType>{$this->processType}</processType>
                <invoiceNo>{$this->invoiceNo}</invoiceNo>
                </PaymentProcessRequest>";
        $psrLogger->debug('paymentRequest: ' . print_r($paymentRequest,1));
        $this->writeToCustomLog('inq_paymentRequest: ' . print_r($paymentRequest,1));
        $jws_token = $this->paymentService->v4_payment_action_jwe($paymentRequest);
        $psrLogger->debug('jws_token: ' . print_r($jws_token,1));
        $this->writeToCustomLog('inq_jws_token: ' . print_r($jws_token,1));

        // Set headers
        $headers = ['content-type' => 'text/plain'];
        // Set timeout options
        $this->curl->setTimeout(60); // Timeout in seconds
        $this->curl->setOption(CURLOPT_CONNECTTIMEOUT_MS, 6000); // Connection timeout in milliseconds

        // Add headers
        foreach ($headers as $key => $value) {
            $this->curl->addHeader($key, $value);
        }
        // Make the POST request
        $this->curl->post($this->url, $jws_token);
        // Get the response
        $response = $this->curl->getBody();

        // $psrLogger->debug('getBody: ' . print_r($this->curl->getBody(),1));

        $psrLogger->debug('response: ' . print_r($response,1));
        $this->writeToCustomLog('inq_response: ' . print_r($response,1));

        $isVerified = $this->paymentService->v4_payment_action_verify($response);
        $psrLogger->debug('isVerified: ' . print_r($isVerified,1));
        $this->writeToCustomLog('inq_isVerified: ' . print_r($isVerified,1));

        if($isVerified){
            $this->writeToCustomLog('inq_paymentProcess - Calling v4_payment_action_decrypt');
            $decrypt_response = $this->paymentService->v4_payment_action_decrypt($isVerified);
            $psrLogger->debug('decrypt_response: ' . print_r($decrypt_response,1));
            $this->writeToCustomLog('inq_decrypt_response: ' . print_r($decrypt_response,1));
            return $decrypt_response;
        } else {
            $this->writeToCustomLog('inq_paymentProcess - isVerified is false, not calling decrypt');
        }
        return null;
    }
    public function cap_setTheParameters($invoiceNo, $amount)
    {
        $this->merchant_id = $this->objConfigSettings["merchantId"];
        $this->secretKey = $this->objConfigSettings["secretKey"];
        $this->processType = "S";
        $this->invoiceNo = $invoiceNo;
        $this->amount = $amount;
        $this->version = "3.4";
        $stringToHash =
            $this->version .
            $this->merchant_id .
            $this->processType .
            $this->invoiceNo .
            $this->amount;
        $this->hash = strtoupper(
            hash_hmac("sha1", $stringToHash, $this->secretKey, false)
        ); //Compute hash value
    }

    public function cap_paymentProcess($invoiceNo, $amount)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        $psrLogger->debug('cap_paymentProcess');

        //set up the parameters
        $this->cap_setTheParameters($invoiceNo, $amount);

        $paymentRequest = "<PaymentProcessRequest>
                <version>{$this->version}</version>
                <merchantID>{$this->merchant_id}</merchantID>
                <processType>{$this->processType}</processType>
                <invoiceNo>{$this->invoiceNo}</invoiceNo>
                <actionAmount>{$amount}</actionAmount>
                </PaymentProcessRequest>";
        $psrLogger->debug('paymentRequest: ' . print_r($paymentRequest,1));
        $jws_token = $this->paymentService->v4_payment_action_jwe($paymentRequest);
        $psrLogger->debug('jws_token: ' . print_r($jws_token,1));

        // Set headers
        $headers = ['content-type' => 'text/plain'];
        // Set timeout options
        $this->curl->setTimeout(60); // Timeout in seconds
        $this->curl->setOption(CURLOPT_CONNECTTIMEOUT_MS, 6000); // Connection timeout in milliseconds

        // Add headers
        foreach ($headers as $key => $value) {
            $this->curl->addHeader($key, $value);
        }
        // Make the POST request
        $this->curl->post($this->url, $jws_token);
        // Get the response
        $response = $this->curl->getBody();

        // $psrLogger->debug('getBody: ' . print_r($this->curl->getBody(),1));

        $psrLogger->debug('response: ' . print_r($response,1));

        $isVerified = $this->paymentService->v4_payment_action_verify($response);
        $psrLogger->debug('isVerified: ' . print_r($isVerified,1));

        if($isVerified){
            $decrypt_response = $this->paymentService->v4_payment_action_decrypt($isVerified);
            $psrLogger->debug('decrypt_response: ' . print_r($decrypt_response,1));
            return $decrypt_response;
        }

        // $http = new \P2c2p\P2c2pPayment\Helper\HTTP();
        // $response = $http->post(
        //     $this->getPaymentGetwayRedirectUrl(),
        //     "paymentRequest=" . $payload
        // );
        // $this->log('response data : ' . base64_decode($response));
        return $response;
    }

    public function ref_setTheParameters($invoiceNo, $amount)
    {
        $this->merchant_id = $this->objConfigSettings["merchantId"];
        $this->secretKey = $this->objConfigSettings["secretKey"];
        $this->processType = "R";
        $this->invoiceNo = $invoiceNo;
        $this->amount = $amount;
        $this->version = "3.4";
        $stringToHash =
            $this->version .
            $this->merchant_id .
            $this->processType .
            $this->invoiceNo .
            $this->amount;
        $this->hash = strtoupper(
            hash_hmac("sha1", $stringToHash, $this->secretKey, false)
        ); //Compute hash value
    }

    public function ref_paymentProcess($invoiceNo, $amount)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        $psrLogger->debug('ref_paymentProcess');
        $this->writeToCustomLog('=== STARTING ref_paymentProcess ===');
        $this->loadsettings();

        //set up the parameters
        $this->ref_setTheParameters($invoiceNo, $amount);

        //Construct request message
        $paymentRequest = "<PaymentProcessRequest>
                <version>{$this->version}</version>
                <merchantID>{$this->merchant_id}</merchantID>
                <processType>{$this->processType}</processType>
                <invoiceNo>{$this->invoiceNo}</invoiceNo>
                <actionAmount>{$amount}</actionAmount>
                <hashValue>{$this->hash}</hashValue>
                </PaymentProcessRequest>";
        $psrLogger->debug('paymentRequest: ' . print_r($paymentRequest,1));
        $this->writeToCustomLog('ref_paymentRequest: ' . print_r($paymentRequest,1));
        $jws_token = $this->paymentService->v4_payment_action_jwe($paymentRequest);
        $psrLogger->debug('jws_token: ' . print_r($jws_token,1));
        $this->writeToCustomLog('ref_jws_token: ' . print_r($jws_token,1));

        // Set headers
        $headers = ['content-type' => 'text/plain'];
        // Set timeout options
        $this->curl->setTimeout(60); // Timeout in seconds
        $this->curl->setOption(CURLOPT_CONNECTTIMEOUT_MS, 6000); // Connection timeout in milliseconds

        // Add headers
        foreach ($headers as $key => $value) {
            $this->curl->addHeader($key, $value);
        }
        // Make the POST request
        $this->curl->post($this->url, $jws_token);
        // Get the response
        $response = $this->curl->getBody();

        // $psrLogger->debug('getBody: ' . print_r($this->curl->getBody(),1));

        $psrLogger->debug('response: ' . print_r($response,1));
        $this->writeToCustomLog('ref_response: ' . print_r($response,1));

        $isVerified = $this->paymentService->v4_payment_action_verify($response);
        $psrLogger->debug('isVerified: ' . print_r($isVerified,1));
        $this->writeToCustomLog('ref_isVerified: ' . print_r($isVerified,1));

        if($isVerified){
            $this->writeToCustomLog('ref_paymentProcess - Calling v4_payment_action_decrypt');
            $decrypt_response = $this->paymentService->v4_payment_action_decrypt($isVerified);
            $psrLogger->debug('decrypt_response: ' . print_r($decrypt_response,1));
            $this->writeToCustomLog('ref_decrypt_response: ' . print_r($decrypt_response,1));
            return $decrypt_response;
        } else {
            $this->writeToCustomLog('ref_paymentProcess - isVerified is false, not calling decrypt');
        }
        return null;
    }

    public function vod_setTheParameters($invoiceNo)
    {
        $this->merchant_id = $this->objConfigSettings["merchantId"];
        $this->secretKey = $this->objConfigSettings["secretKey"];
        $this->processType = "V";
        $this->invoiceNo = $invoiceNo;
        $this->version = "3.4";
        $stringToHash =
            $this->version .
            $this->merchant_id .
            $this->processType .
            $this->invoiceNo;
        $this->hash = strtoupper(
            hash_hmac("sha1", $stringToHash, $this->secretKey, false)
        ); //Compute hash value
    }

    public function vod_paymentProcess($invoiceNo)
    {
        $psrLogger = ObjectManager::getInstance()->get(PsrLoggerInterface::class); 
        $psrLogger->debug('vod_paymentProcess');
        $this->writeToCustomLog('=== STARTING vod_paymentProcess ===');
        $this->loadsettings();

        //set up the parameters
        $this->vod_setTheParameters($invoiceNo);

        //Construct request message
        $paymentRequest = "<PaymentProcessRequest>
                <version>{$this->version}</version>
                <merchantID>{$this->merchant_id}</merchantID>
                <processType>{$this->processType}</processType>
                <invoiceNo>{$this->invoiceNo}</invoiceNo>
                <hashValue>{$this->hash}</hashValue>
                </PaymentProcessRequest>";
        $psrLogger->debug('paymentRequest: ' . print_r($paymentRequest,1));
        $this->writeToCustomLog('vod_paymentRequest: ' . print_r($paymentRequest,1));
        $jws_token = $this->paymentService->v4_payment_action_jwe($paymentRequest);
        $psrLogger->debug('jws_token: ' . print_r($jws_token,1));
        $this->writeToCustomLog('vod_jws_token: ' . print_r($jws_token,1));

        // Set headers
        $headers = ['content-type' => 'text/plain'];
        // Set timeout options
        $this->curl->setTimeout(60); // Timeout in seconds
        $this->curl->setOption(CURLOPT_CONNECTTIMEOUT_MS, 6000); // Connection timeout in milliseconds

        // Add headers
        foreach ($headers as $key => $value) {
            $this->curl->addHeader($key, $value);
        }
        // Make the POST request
        $this->curl->post($this->url, $jws_token);
        // Get the response
        $response = $this->curl->getBody();

        // $psrLogger->debug('getBody: ' . print_r($this->curl->getBody(),1));

        $psrLogger->debug('response: ' . print_r($response,1));
        $this->writeToCustomLog('vod_response: ' . print_r($response,1));

        $isVerified = $this->paymentService->v4_payment_action_verify($response);
        $psrLogger->debug('isVerified: ' . print_r($isVerified,1));
        $this->writeToCustomLog('vod_isVerified: ' . print_r($isVerified,1));

        if($isVerified){
            $this->writeToCustomLog('vod_paymentProcess - Calling v4_payment_action_decrypt');
            $decrypt_response = $this->paymentService->v4_payment_action_decrypt($isVerified);
            $psrLogger->debug('decrypt_response: ' . print_r($decrypt_response,1));
            $this->writeToCustomLog('vod_decrypt_response: ' . print_r($decrypt_response,1));
            return $decrypt_response;
        } else {
            $this->writeToCustomLog('vod_paymentProcess - isVerified is false, not calling decrypt');
        }
        return null;
    }
}
